<?php
require_once __DIR__ . '/Database.php';

class Payroll {
    private $conn;
    private $table = 'payroll_transactions';
    private $config_table = 'payroll_config';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    private function getConfig() {
        $query = "SELECT * FROM " . $this->config_table;
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_KEY_PAIR); // config_key => config_value
        return $results;
    }

    public function generateFinalSalary($month, $year) {
        // 1. Get Config
        $config = $this->getConfig();
        $pf_percent = isset($config['pf_employee_percent']) ? $config['pf_employee_percent'] : 10;
        
        // 2. Get Active Employees
        $query = "SELECT id, basic_salary FROM employees WHERE status = 'Regular' OR status = 'Probation'";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $employees = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // 3. Process Each Employee
        foreach ($employees as $emp) {
            $basic = $emp['basic_salary'];
            
            // Allowances
            // Note: Fixed allowances from config or percentage
            // Assuming config values are absolute amounts if not specified as percent, 
            // but schema showed '75' for house_rent_percent.
            
            $house_rent = $basic * ($config['house_rent_percent'] / 100);
            $medical = $config['medical_allowance'];
            $conveyance = $config['conveyance_allowance'];
            $special = $basic * 0.05; // 5% Special Allowance as per template header
            
            // Variable Allowance (Needs to be input separately, setting 0 for auto-process if not exists)
            // Ideally should check if record exists and preserve variable allowance?
            // "This will overwrite previous values" warning in template suggests full overwrite or update.
            // Let's check if record exists to fetch current variable allowance if we don't want to reset it.
            // But usually "Process" might mean calculating the fixed parts. 
            // Let's set variable to 0 for now as safe default or existing if I could but I'll stick to 0 for fresh generation.
            $variable = 0; 

            // Deductions
            $pf = $basic * ($pf_percent / 100);
            $other_deduct = 0;

            $gross = $basic + $house_rent + $medical + $conveyance + $special + $variable;
            $net = $gross - ($pf + $other_deduct);

            // 4. Insert or Update
            // Check if exists
            $check = "SELECT id FROM " . $this->table . " WHERE employee_id = :eid AND month = :m AND year = :y";
            $stmtC = $this->conn->prepare($check);
            $stmtC->execute([':eid' => $emp['id'], ':m' => $month, ':y' => $year]);
            
            if ($stmtC->rowCount() > 0) {
                 // Update
                 $uQ = "UPDATE " . $this->table . " SET 
                        basic_salary=:basic, house_rent=:hr, medical=:med, conveyance=:conv, 
                        special_allowance=:special, pf_deduction=:pf, net_pay=:net 
                        WHERE employee_id=:eid AND month=:m AND year=:y";
                 $stmtU = $this->conn->prepare($uQ);
                 $stmtU->execute([
                     ':basic' => $basic, ':hr' => $house_rent, ':med' => $medical, ':conv' => $conveyance,
                     ':special' => $special, ':pf' => $pf, ':net' => $net,
                     ':eid' => $emp['id'], ':m' => $month, ':y' => $year
                 ]);
            } else {
                // Insert
                $iQ = "INSERT INTO " . $this->table . " 
                       (employee_id, month, year, basic_salary, house_rent, medical, conveyance, special_allowance, variable_allowance, pf_deduction, other_deductions, net_pay, status)
                       VALUES (:eid, :m, :y, :basic, :hr, :med, :conv, :special, :var, :pf, :other, :net, 'Draft')";
                $stmtI = $this->conn->prepare($iQ);
                $stmtI->execute([
                     ':eid' => $emp['id'], ':m' => $month, ':y' => $year,
                     ':basic' => $basic, ':hr' => $house_rent, ':med' => $medical, ':conv' => $conveyance,
                     ':special' => $special, ':var' => $variable, ':pf' => $pf, ':other' => $other_deduct, ':net' => $net
                 ]);
            }
        }
    }

    public function getSalarySheet($month, $year) {
        $query = "SELECT p.*, e.full_name, e.employee_id as emp_code, e.designation 
                  FROM " . $this->table . " p
                  JOIN employees e ON p.employee_id = e.id
                  WHERE p.month = :m AND p.year = :y
                  ORDER BY e.id ASC";
        $stmt = $this->conn->prepare($query);
        $stmt->execute([':m' => $month, ':y' => $year]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
